package elaprendiz.graficos.objetos;

import java.awt.*;

/**
 * An abstract class for displaying graphics objects. A graphics
 * object is loosely defined as any object that can be drawn on
 * a graphics context. A ObjetosGraficos is centered upon an origen 
 * which is an instance of Point. This origen is passed to the 
 * ObjetosGraficos during construction and represents the (x,y) 
 * location of the object within the graphics context.
 * <P>
 * A ObjetosGraficos guarantees only three methods
 * <BLOCKQUOTE><UL>
 * <LI><CODE>getOrigin()</CODE>
 * </UL></BLOCKQUOTE>
 * which is defined by ObjetosGraficos, and two abstract methods
 * <BLOCKQUOTE><UL>
 * <LI><CODE>draw(Graphics g)</CODE>
 * <LI><CODE>size()</CODE>
 * </UL></BLOCKQUOTE>
 * Any other functions as well as all abstract methods are the 
 * responsibility of ObjetosGraficos subclasses.
 * <P>
 * Here is an example (albeit, inefficient) of how a ObjetosGraficos
 * might be used:
 * <P>
 * <PRE>
 * ObjetosGraficos object;
 * <BR>
 *   . . .
 * <BR>
 * public void paint(Graphics g) {
 *   Dimension d = size();
 * <BT>
 *   Point gPoint = object.getOrigin();
 *   Dimension gSize = object.size();
 *
 *   if( ( ( gPoint.x - gSize.width/2) > 0) &&
 *       ( ( gPoint.x + gSize.width/2) < d.width) &&
 *       ( ( gPoint.y - gSize.height/2) > 0) &&
 *       ( ( gPoint.y + gSize.height/2) < d.height))
 *     object.draw(g);
 * }
 * </PRE>
 * <P>
 * @version 1.1 March 27, 1997
 * @author  John Kloss
 */

public abstract class ObjetosGraficos {

  /** The origen of the ObjetosGraficos. The ObjetosGraficos is
   *  centered upon this point. 
   */
  protected Point origen;


  /** Constructor. This method instantiates the ObjetosGraficos
   *  and sets its origen to the origen. This method is protected 
   *  so that only subclasses of ObjetosGraficos can call it. 
   *
   * @param origen          The Point to center the ObjetosGraficos
   *                        upon.
   */
  protected ObjetosGraficos (Point origin) {
    this.origen = origin;
  }

  /** Return the origen of the ObjetosGraficos. The Point object 
   *  returned is a new instance of Point. This insures that the 
   *  ObjetosGraficos's origen is not mistakenly changed by this
   *  call.
   * 
   * @return                The location of the ObjetosGraficos
   *                        in a graphics context.
   */
  public Point origin() {
    return new Point(origen.x, origen.y);
  }

  /** Abstract method which draws the ObjetosGraficos. It is the
   *  responsibility of ObjetosGraficos's subclasses to implement
   *  this method.
   *
   * @param g                 The graphics context upon which the
   *                          ObjetosGraficos will draw itself.
   */
  public abstract void draw(Graphics g);

  /** Abstract method which returns the size of the ObjetosGraficos.
   *  It is the responsibility of ObjetosGraficos's subclasses to
   *  implement this method.
   *
   * @return                  A Dimension object the size of the 
   *                          ObjetosGraficos
   */
  public abstract Dimension size();
}


